#include <linux/irq.h>
#include <linux/cdev.h>
#include <linux/module.h>
#include <linux/kdev_t.h>
#include <linux/poll.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/timer.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/fs.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/jiffies.h>
#include <asm/uaccess.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,33)
#include <generated/autoconf.h>
#else
#include <linux/autoconf.h>
#endif
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <dvl_features.h>
#ifdef IXP_GPIO
#include <asm/hardware.h>
#endif
#ifdef LPC32XX_GPIO
#include <arch/arm/mach-lpc32xx/include/mach/gpio.h>
#include <arch/arm/mach-lpc32xx/sys-lpc32xx.h>

#define GPIO_OUTP_SET	0x04
#define GPIO_OUTP_CLR	0x08
#define GPIO_DIR_SET	0x10
#define GPIO_OUTP_STATE 0x0C

#define gpio_writel(val, reg)	__raw_writel(val, GPIO_IOBASE + (reg))
#define gpio_readl(reg)	__raw_readl(GPIO_IOBASE + (reg))
#endif

#ifdef AR934X_HW_WDT
#include <ar71xx_regs.h>
#endif

#ifdef _DVL_DISABLE_WATCHDOG
#include "disableHeartbeat.h"
#endif

#define MODULE_NAME  "dvlheartbeat"
MODULE_LICENSE("GPL");

#ifdef BOARD_934X_HomeControl
#include <atheros.h>
#define ATH_WD_ACT_NONE			0u /* No Action */
#define ATH_WD_ACT_RESET		3u /* Full Chip Reset */
#endif


static void start_timer(void);

static struct timer_list wdt;

#define TIMER_MARGIN (5*60)            /* Default is 5 minutes */
static int soft_margin = TIMER_MARGIN; /* in seconds */
static unsigned long driver_open, orphan_timer;
static int counter = TIMER_MARGIN;

static unsigned int dvlheartbeat_major = 244; // 0=use automatic majornumber, not static
                                              // can be read from /proc/devices
static unsigned int dvlheartbeat_minor = 0;   // use automatic minornumber

struct dvlheartbeat_dev {
    struct semaphore sem;
    struct cdev cdev;         /* character device */
};

static dev_t dev_id;
static struct dvlheartbeat_dev global_dev;

static int wdt_keepalive(void);

/*
 *  timer operations
 */

#ifdef LPC32XX_GPIO
static int pin;
static int gpiostate;
static int init_gpiostate = 1;
#endif

static void wdt_handler(unsigned long unused_arg)
{
#ifdef IXP_GPIO
    static int gpiostate = IXP4XX_GPIO_LOW;
    gpiostate = (IXP4XX_GPIO_LOW==gpiostate ) ? IXP4XX_GPIO_HIGH : IXP4XX_GPIO_LOW;  
    gpio_line_set(DVLHOST_WDI, gpiostate);
#endif

#ifdef LPC32XX_GPIO
    if(init_gpiostate)
    {
	    init_gpiostate = 0;
	    gpiostate = (gpio_readl(GPIO_OUTP_STATE)&(1<<(pin-67)));
	    gpiostate = (0==gpiostate ) ? 0 : 1;
    }
    gpiostate = (0==gpiostate ) ? 1 : 0;

    gpio_writel(1 << (pin-67), (gpiostate ? GPIO_OUTP_SET : GPIO_OUTP_CLR));
#endif

#ifdef AR934X_HW_WDT
    static volatile long *wdt_tmr = KSEG1ADDR(AR71XX_RESET_BASE + AR71XX_RESET_REG_WDOG);
    *wdt_tmr = 0xffffffff;
#endif

#ifdef BOARD_934X_HomeControl 
    uint32_t usec = 5 * USEC_PER_SEC;  // 5s HW watchdog timer

#ifdef CONFIG_MACH_AR934x
	usec = usec * (ath_ref_freq / USEC_PER_SEC);
#else
	usec = usec * (ath_ahb_freq / USEC_PER_SEC);
#endif

#ifndef _DVL_DISABLE_WATCHDOG
    ath_reg_wr(ATH_WATCHDOG_TMR_CONTROL, ATH_WD_ACT_NONE);
    ath_reg_wr(ATH_WATCHDOG_TMR, usec);
    ath_reg_wr(ATH_WATCHDOG_TMR_CONTROL, ATH_WD_ACT_RESET);
#endif

#endif
    start_timer();
}

#ifdef _DVL_DISABLE_WATCHDOG
static int disableWatchdog = 0;
#endif
static void start_timer(void)
{
    init_timer(&wdt);
    wdt.data = 0;
    wdt.function = wdt_handler;
#ifdef LPC32XX_GPIO
    wdt.expires = jiffies + ((4*(HZ))/10); //400 milliseconds
#else
    wdt.expires = jiffies + HZ; //1000 milliseconds
#endif
    counter = counter - 1;

#ifdef _DVL_DISABLE_WATCHDOG
    if (counter % 10 == 0)
    {
        disableWatchdog = 0;
        if (shouldWatchdogDisabled() && (counter-1 != soft_margin))
        {
            disableWatchdog = 1;
            wdt_keepalive();
        }

        if (disableWatchdogStatusChange())
        {
            printk(KERN_INFO "%s: watchdog has been %s due softirq load\n",
                             MODULE_NAME,disableWatchdog ? "disabled" : "enabled");
        }
    }
#endif

    if(counter <= 10)
    {
	    printk(KERN_CRIT "Watchdog counter %d\n",counter);
    }

    if(counter <= 0)
    {
        printk(KERN_CRIT "Watchdog initiates system reboot.\n");
        emergency_restart();
        printk(KERN_CRIT "Reboot didn't ?????\n");
    }
    else
    {
        add_timer(&wdt);
    }
}

/*
 *  Watchdog operations
 */
static int wdt_keepalive(void)
{
    counter = soft_margin;
    return 0;
}

static int wdt_stop(void)
{
    // todo 
    return 0;
}

static int wdt_set_heartbeat(int t)
{
    if ((t < 0x0001) || (t > 0xFFFF))
        return -EINVAL;

    soft_margin = t;
    return 0;
}


/*
 *  /dev/MODULE_NAME handling
 */

static int wdt_open(struct inode *inode, struct file *file)
{
    int rv;
    if (test_and_set_bit(0, &driver_open))
        return -EBUSY;
    if (!test_and_clear_bit(0, &orphan_timer))
        __module_get(THIS_MODULE);

    rv = nonseekable_open(inode, file);
    return rv;
}

static int wdt_release(struct inode *unused_inode, struct file *unused_file)
{
    int rv = 0;
    set_bit(0, &orphan_timer);
    clear_bit(0, &driver_open);
    return rv;
}


static ssize_t wdt_write(struct file *file, const char __user *data, size_t len, loff_t *ppos)
{
    /*
     *  Refresh the timer.
     */
    if(len) {
        wdt_keepalive();
    }
    return len;
}

/*
 * Called if read() is called
 */
static ssize_t wdt_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos) {
    ssize_t retval = 0;

	char *s = (char *) kzalloc((16), GFP_KERNEL);
	if (s == NULL)
	{
		printk(KERN_ERR "unable to allocate memory\n");
	}
    else
    {
		char line[16];
		snprintf(line, sizeof(line), "%d\n", counter);
		strcat(s, line);

        if (copy_to_user(buf, s, strlen(s)))
            retval = -EFAULT;
        else
            retval = (ssize_t) strlen(s);
        
        kfree(s);
    }

    return retval;
}

//signature handling for different kernel versions
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
//.ioctl was removed from kernel >= version 2.6.36 and replaced by
//.unlocked_ioctl and .compat_ioctl
static long wdt_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
#else
static int  wdt_ioctl(struct inode *inode, struct file *file,
                      unsigned int cmd, unsigned long arg)
#endif
{
    void __user *argp = (void __user *)arg;
    int __user *p = argp;
    int new_margin;
    static struct watchdog_info ident = {
        .options =  WDIOF_SETTIMEOUT |
        WDIOF_KEEPALIVEPING |
        WDIOF_MAGICCLOSE,
        .firmware_version = 0,
        .identity = MODULE_NAME,
    };
    switch (cmd) {
        default:
            return -ENOTTY;
        case WDIOC_GETSUPPORT:
            return copy_to_user(argp, &ident,
                                sizeof(ident)) ? -EFAULT : 0;
        case WDIOC_GETSTATUS:
        case WDIOC_GETBOOTSTATUS:
            return put_user(0, p);
        case WDIOC_KEEPALIVE:
        {
            int rv;
            rv = wdt_keepalive();
            return rv;
        }
        case WDIOC_SETTIMEOUT:
            if (get_user(new_margin, p))
                return -EFAULT;
            if (wdt_set_heartbeat(new_margin))
                return -EINVAL;
            wdt_keepalive();
            /* Fall */
        case WDIOC_GETTIMEOUT:
            return put_user(soft_margin, p);
    }
}

/*
 *  Notifier for system down
 */

static int wdt_notify_sys(struct notifier_block *this, unsigned long code,
                          void *unused)
{
    if(code==SYS_DOWN || code==SYS_HALT) {
        /* Turn the WDT off */
        wdt_stop();
    }
    return NOTIFY_DONE;
}


/*
 *  Kernel Interfaces
 */

static const struct file_operations wdt_fops = {
    .owner          = THIS_MODULE,
    .llseek         = no_llseek,
    .write          = wdt_write,
    .read           = wdt_read,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
    //.ioctl was removed from kernel >= version 2.6.36 and replaced by
    //.unlocked_ioctl and .compat_ioctl
    .unlocked_ioctl = wdt_ioctl,
#else
    .ioctl          = wdt_ioctl,
#endif
    .open           = wdt_open,
    .release        = wdt_release,
};

static struct notifier_block wdt_notifier = {
    .notifier_call  = wdt_notify_sys,
};


/*
 * Setup character devices
 */
static void dvlheartbeat_setup_cdev(struct dvlheartbeat_dev *dev, unsigned int index) {
    int err;
    unsigned int devno = MKDEV(dvlheartbeat_major, dvlheartbeat_minor + index);

    cdev_init(&dev->cdev, &wdt_fops);
    dev->cdev.owner = THIS_MODULE;
    dev->cdev.ops = &wdt_fops;
    err = cdev_add(&dev->cdev, devno, 1);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,37)
    //does the same like init_MUTEX(), but dissapeared in kernel 2.6.37
    sema_init(&dev->sem,1);
#else
    //more readable then sema_init()
    init_MUTEX(&dev->sem);
#endif

    if (err)
        printk(KERN_DEBUG "Error %d adding %s%d",err,MODULE_NAME,index);
}

/*
 *  module
 */

static int __init init_mod(void)
{
    int ret;    

    dev_id = MKDEV(dvlheartbeat_major, dvlheartbeat_minor);
    ret = register_chrdev_region(dev_id,1,MODULE_NAME);
    if (ret < 0) {
        printk(KERN_WARNING "%s, can't get major %d\n",MODULE_NAME, dvlheartbeat_major);
        return ret;
    }
    dvlheartbeat_setup_cdev(&global_dev, 0);

    ret = register_reboot_notifier(&wdt_notifier);
    if (ret) {
        printk (KERN_ERR "cannot register reboot notifier (err=%d)\n",
                ret);
        cdev_del(&global_dev.cdev);
        unregister_chrdev_region(dev_id,1);
        return ret;
    }

#ifdef LPC32XX_GPIO
    switch(system_rev) {
	    case 2:
		    gpio_writel((1<<25),GPIO_DIR_SET);
		    pin = 92;
		    break;
	    default:
		    pin = 78;
		    break;
    }
#endif

#ifdef IXP_GPIO
    gpio_line_config(DVLHOST_WD_LED_ENABLE, IXP4XX_GPIO_OUT);
    gpio_line_set(DVLHOST_WD_LED_ENABLE, IXP4XX_GPIO_LOW);
    gpio_line_config(DVLHOST_WDI, IXP4XX_GPIO_OUT);
    gpio_line_set(DVLHOST_WDI, gpiostate);
#endif
    start_timer();

    printk(KERN_INFO "Module %s initialized\n", MODULE_NAME);   
    return 0;
}

static void __exit exit_mod(void)
{
#ifdef BOARD_934X_HomeControl
    ath_reg_wr(ATH_WATCHDOG_TMR_CONTROL, ATH_WD_ACT_NONE);  // remove HW WD
#endif

    unregister_reboot_notifier(&wdt_notifier);
    cdev_del(&global_dev.cdev);
    unregister_chrdev_region(dev_id,1);

    del_timer_sync(&wdt); 
#ifdef IXP_GPIO
    gpio_line_config(DVLHOST_WDI, IXP4XX_GPIO_IN);
    gpio_line_config(DVLHOST_WD_LED_ENABLE, IXP4XX_GPIO_IN);
#endif
    printk(KERN_INFO "Module %s removed\n", MODULE_NAME);
}

MODULE_AUTHOR("Christian Taedcke");
MODULE_DESCRIPTION("devolo heartbeat module");

module_init(init_mod);
module_exit(exit_mod);

//EXPORT_NO_SYMBOLS;
