
#include <linux/autoconf.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/leds.h>

#include "dvlled_mod.h"

#define MODULE_NAME  "dvlled"
MODULE_LICENSE("GPL");

//
// Prototypes
//

static int dvlled_probe(struct platform_device *pdev);
static int dvlled_remove(struct platform_device *pdev);
static void dvlled_brightness_set(struct led_classdev *pled, enum led_brightness value);

//
// Device and driver structures
//

struct platform_device dvl_platform_device =
{
	.name              = LED_DEVICE_NAME,
	.id                = -1,
	.dev.platform_data = &global_leds,
	.num_resources     = 0,
	.resource          = NULL,
};

struct platform_driver dvl_platform_driver =
{
	.probe  = dvlled_probe,
	.remove = dvlled_remove,
	.driver =
	{
		.name = LED_DEVICE_NAME,
	},
};

struct dvlled_device
{
	struct led_classdev ancestor;
	
	char name[128];
	int index;
};

static struct dvlled_device *led_devices = NULL;

//
// Helper functions
//

static int num_leds(void)
{
	int rv = 0;
	struct dvlled_entry *l;
	
	for (l = global_leds, rv = 0; l->type != TYPE_NONE; l++, rv++)
	{
	}
	
	return rv;
}

static void led_set_value(int index, int value)
{
	struct dvlled_entry *l = &(global_leds[index]);
	
	int value_on  = (l->polarity == 0) ? 1 : 0;
	int value_off = 1 - value_on;
	
	if (l->type == TYPE_LED)
	{
		if (l->line1 >= 0)
		{
			if (value != 0)
				dvl_led_set(l->line1, value_on);
			else
				dvl_led_set(l->line1, value_off);
		}
	}
	else if (l->type == TYPE_DUAL_LED)
	{
		if (l->line1 >= 0)
		{
			if ((value & 1) != 0)
				dvl_led_set(l->line1, value_on);
			else
				dvl_led_set(l->line1, value_off);
		}
		
		if (l->line2 >= 0)
		{
			if ((value & 2) != 0)
				dvl_led_set(l->line2, value_on);
			else
				dvl_led_set(l->line2, value_off);
		}
	}
}

//
// Device callbacks
//

static int dvlled_probe(struct platform_device *pdev)
{
	int num = num_leds();
	int i;

	dvl_led_probe(pdev);

	led_devices = kzalloc(num * sizeof(struct dvlled_device), GFP_KERNEL);

	for (i = 0; i < num; i++)
	{
		struct dvlled_entry *l = &(global_leds[i]);
		int rc;

		snprintf(led_devices[i].name, sizeof(led_devices[i].name), "%s", global_leds[i].name);
		led_devices[i].index = i;

		led_devices[i].ancestor.name = led_devices[i].name;
		led_devices[i].ancestor.flags = 0;
		led_devices[i].ancestor.brightness = LED_OFF;
		led_devices[i].ancestor.brightness_set = dvlled_brightness_set;

		rc = led_classdev_register(&pdev->dev, &led_devices[i].ancestor);
		if (rc < 0)
		{
			led_devices[i].ancestor.name = 0;
			dvlled_remove(pdev);
			return rc;
		}
		
		if (l->type == TYPE_LED)
		{
			if (global_leds[i].line1 >= 0)
				dvl_led_config_out(global_leds[i].line1);
		}
		else if (l->type == TYPE_DUAL_LED)
		{
			if (global_leds[i].line1 >= 0)
				dvl_led_config_out(global_leds[i].line1);
			if (global_leds[i].line2 >= 0)
				dvl_led_config_out(global_leds[i].line2);
		}
		
		led_set_value(i, 0);
	}

	return 0;
}

static int dvlled_remove(struct platform_device *pdev)
{
	int num = num_leds();
	int i;

	for(i = 0; i < num; i++)
	{
		if (led_devices[i].ancestor.name != 0)
		{
			led_classdev_unregister(&led_devices[i].ancestor);
			led_devices[i].ancestor.name = 0;
		}
	}

	kfree(led_devices);
	led_devices = NULL;

	dvl_led_remove(pdev);

	return 0;
}

static void dvlled_brightness_set(struct led_classdev *pled, enum led_brightness value)
{
	const struct dvlled_device *const dev = container_of(pled, struct dvlled_device, ancestor);

	led_set_value(dev->index, (int) value);
}

//
// Init/Exit
//

static int __init init_mod(void)
{
	int rc = 0;
	
	dvl_led_init();
	
	platform_device_register(&dvl_platform_device);
	rc = platform_driver_register(&dvl_platform_driver);

	if (rc >= 0)
	{
		printk(KERN_INFO "Module %s initialized\n", MODULE_NAME);   
	}
	return rc;
}

static void __exit exit_mod(void)
{
	platform_driver_unregister(&dvl_platform_driver);
	platform_device_unregister(&dvl_platform_device);
	
	printk(KERN_INFO "Module %s removed\n", MODULE_NAME);
}

MODULE_AUTHOR("Christian Petry");
MODULE_DESCRIPTION("LED module");

module_init(init_mod);
module_exit(exit_mod);
