
#include "dvlbutton_mod.h"

#define MODULE_NAME  "dvlbutton"
MODULE_LICENSE("GPL");

struct dvlbutton_context *global_contexts = NULL;

static unsigned int dvlbutton_major = 245; // 0=use automatic majornumber, not static
                                           // can be read from /proc/devices
static unsigned int dvlbutton_minor = 0;   // use automatic minornumber

static dev_t dev_id;
static struct dvlbutton_dev global_dev;

//
// Helper functions
//
 
static int num_buttons(void)
{
	int rv = 0;
	dvlbutton_entry_t *b;
	
	for (b = global_buttons, rv = 0; b->type != TYPE_NONE; b++, rv++)
	{
	}
	
	return rv;
}

static int button_idx(char id)
{
    int rv = 0;
    dvlbutton_entry_t *b;
    for (b = global_buttons, rv = 0; b->type != TYPE_NONE; b++, rv++)
    {
        if (b->type == TYPE_BUTTON && b->c1 == id) return rv;
    }
    return -1;
}

#if 1 //this is for sysfs
static int button_idx_by_name(char *btname)
{
    int rv = 0;
    dvlbutton_entry_t *b;
    for (b = global_buttons, rv = 0; b->type != TYPE_NONE; b++, rv++)
    {
        if (strcmp(btname,b->name) == 0)
        return rv;
    }
    return -1;
}
#endif

static unsigned long timespec_diff(const struct timespec *t1, const struct timespec *t2)
{
	unsigned long s1 = t1->tv_sec;
	unsigned long s2 = t2->tv_sec;
	unsigned long m1 = t1->tv_nsec / 1000000L;
	unsigned long m2 = t2->tv_nsec / 1000000L;

	unsigned long md = (m2 + (s2 - s1) * 1000L) - m1;
	return md;
}

#define BOUNCE_TIME_MS 20
static int in_bounce_time(const struct timespec *t1, const struct timespec *t2)
{
	unsigned long md = timespec_diff(t1, t2);

	if (md <= BOUNCE_TIME_MS)
		return 1;
	return 0;
}

//
// Output to the character device
//

static char output_queue[64];
static unsigned int write_pos = 0;
static unsigned int file_offset = 0;

static void move_queue(void)
{
	unsigned int offset = sizeof(output_queue) / 2;
	unsigned int i;
	
	for (i = 0; i < sizeof(output_queue) - offset; i++)
		output_queue[i] = output_queue[i + offset];
		
	write_pos -= offset;
	file_offset += offset;
}

static void enqueue_char(struct dvlbutton_dev *dev, char c)
{
	output_queue[write_pos++] = c;
	if (write_pos >= sizeof(output_queue))
		move_queue();
	wake_up_interruptible_all(&dev->inq);
}
 
static ssize_t emit_char(char __user *buf, unsigned int offset)
{
	ssize_t retval = 0;

	if (copy_to_user(buf, &(output_queue[offset]), sizeof(char)))
		retval = -EFAULT;
	else
		retval = (ssize_t) sizeof(char);
    
	return retval;
}

#define CHARS_PER_BUTTON 128
static ssize_t emit_states(char __user *buf)
{
	ssize_t retval = 0;
	int i;
	int num = num_buttons();
	
	char *s = (char *) kzalloc((num + 1) * CHARS_PER_BUTTON, GFP_KERNEL);
	if (s == NULL)
	{
		printk(KERN_ERR "unable to allocate memory\n");
		return retval;
	}

	for (i = 0; i < num; i++)
	{
		char line[CHARS_PER_BUTTON + 1];
		char c = DVLBUTTON_NONE;
		dvlbutton_entry_t *b = &(global_buttons[i]);
		
		int value = 0;
		if (b->gpio >= 0)
			dvl_gpio_line_get(b->gpio, &value);
		
		value = !!value;
		if (b->polarity)
			value = !value;
			
		if (b->type == TYPE_BUTTON || b->type == TYPE_BUTTON_SHORTLONG)
		{
			c = value ? (b->c1) : DVLBUTTON_NONE;
		}
		else if (b->type == TYPE_SWITCH)
		{
			c = value ? (b->c1) : (b->c2);
		}
		
		snprintf(line, sizeof(line), "%c - %s\n", c, b->name);
		strcat(s, line);
	}
	
	if (copy_to_user(buf, s, strlen(s)))
		retval = -EFAULT;
	else
		retval = (ssize_t) strlen(s);
	
	kfree(s);
	return retval;
}

//
// Timers and IRQ handlers
// 

/*
 * Called if timer is expired
 */
static void timer_handler(unsigned long arg) {
	int button = (int) arg;
    global_contexts[button].is_long = 1; /* remember until release of button */
   	enqueue_char(global_contexts[button].dev, global_buttons[button].c2);
}


/*
 * Called if button is released
 */
static irqreturn_t shortlong_stop_handler(int button, void *unused_ctx)
{
	del_timer_sync(&(global_contexts[button].timer));
	if (global_contexts[button].is_long == 0)
	{
		struct timespec now;
		ktime_get_ts(&now);
		
		if (timespec_diff(&(global_contexts[button].last_time), &now) <= global_buttons[button].param1)
		{
			enqueue_char(global_contexts[button].dev, global_buttons[button].c1);
		}
	}
	else
	{
		global_contexts[button].is_long = 0;
	}
	return IRQ_HANDLED;
}

/*
 * Called if button is pressed
 */
static irqreturn_t shortlong_start_handler(int button, void *unused_ctx) {
	init_timer(&(global_contexts[button].timer));
	global_contexts[button].timer.data = (unsigned long) button;
	global_contexts[button].timer.function = timer_handler;
	global_contexts[button].timer.expires = jiffies + global_buttons[button].param2 * (unsigned int) HZ / 1000;
	add_timer(&(global_contexts[button].timer));
	return IRQ_HANDLED;
}

/*
 * Called if button is released or pressed
 */
static irqreturn_t irq_handler(int irq, void* ctx) 
{
	irqreturn_t rv = IRQ_HANDLED;
	int button = -1, change = -1;
	struct timespec now;

	dvl_gpio_change_get(irq, &button, &change);
	dvl_gpio_clear();

	if (button >= 0)
	{
		dvlbutton_entry_t *b = &(global_buttons[button]);
		ktime_get_ts(&now);

		change = !!change; 	// force 'change' to a boolean 1 or 0 value 
		if (b->polarity == POL_LA)
			change = !change;
		
		if (change != global_contexts[button].last_change)
		{
			if (!in_bounce_time(&(global_contexts[button].last_time), &now))
			{
				if (b->type == TYPE_BUTTON)
				{
					if (change) // button pressed
					{
					}
					else // button released
					{
						enqueue_char(global_contexts[button].dev, b->c1);
					}
				}
				else if (b->type == TYPE_BUTTON_TWOWAY)
                {
				    if (change) // button pressed
                    {
				        enqueue_char(global_contexts[button].dev, b->c1);
                    }
                    else // button released
                    {
                        enqueue_char(global_contexts[button].dev, b->c2);
                    }
                }
				else if (b->type == TYPE_BUTTON_SHORTLONG)
				{
					if (change) // button pressed
					{
						rv = shortlong_start_handler(button, ctx);
					}
					else // button released
					{
						rv = shortlong_stop_handler(button, ctx);
					}
				}
				else if (b->type == TYPE_SWITCH)
				{
					if (change) // switch enabled
					{
						enqueue_char(global_contexts[button].dev, b->c1);
					}
					else // switch disabled
					{
						enqueue_char(global_contexts[button].dev, b->c2);
					}
				}
				
				global_contexts[button].last_change = change;
				memcpy(&(global_contexts[button].last_time), &now, sizeof(struct timespec));
			}
		}
	}

	return rv;
}

//
// sysfs connection
//
#ifdef _DVL_ENABLE_BUTTON_MAPPING
//this buffer is associated to sysfs and contains the actual button/time/function mapping
static char sysfs_buffer[SYSFS_MAX_SIZE] = {0};

//helper for actual size of sysfs_buffer
static unsigned long sysfs_buffer_size = 0;

struct elem_s {
	char key[MAX_KEY_VALUE_SIZE];
	char value[MAX_KEY_VALUE_SIZE];
	struct list_head list;
};

struct parsed_data_s {
	char name[128];
	int gl_btn_index;
	struct elem_s elem;
};

//pointer to structure of new sysfs class buttons
static struct class  *buttons_class = NULL;

//here: pointer to character device /dev/dvlbuttons
static struct device *buttons_device = NULL;

//Helper function to reset the whole buffer
static void clear_sysfs_buffer(char *to_buffer)
{
	memset(sysfs_buffer,0,SYSFS_MAX_SIZE);
	sysfs_buffer_size=0;
}

static void add_str_to_sysfs_buffer(char *to_buffer, char *str)
{
	int len = strlen(str);
	if (to_buffer[0]=='\0') //buffer empty condition
	{
		strncpy(to_buffer,str,len);
	}
	else
	{
		strncat(sysfs_buffer,str,len);
	}
	sysfs_buffer_size+=len;
}

static void add_parameter_to_sysfs_buffer(char *to_buffer, char *parameter)
{
	add_str_to_sysfs_buffer(to_buffer,parameter);
	add_str_to_sysfs_buffer(to_buffer,","); //parameters are divided by comma
}


static void update_global_button_list_for_sysfs(void)
{
	clear_sysfs_buffer(sysfs_buffer);
	int rv = 0;
	dvlbutton_entry_t *b;
	char params [16], character[2] = "\0\0";

	for (b = global_buttons, rv = 0; b->type != TYPE_NONE; b++, rv++)
	{
		char **valid_btn = valid_buttons;
		while (*valid_btn != NULL)
		{
			if (strcmp(*valid_btn,b->name) == 0)
			{
				//handle the beginning of the entry and the name of the given button
				add_str_to_sysfs_buffer(sysfs_buffer,"[");
				add_parameter_to_sysfs_buffer(sysfs_buffer,b->name);

				//handle the first character which is associated by a short push
				character[0] = b->c1;
				add_parameter_to_sysfs_buffer(sysfs_buffer,&character);

				//handle the second character which is associated by a long push
				character[0] = b->c2;
				add_parameter_to_sysfs_buffer(sysfs_buffer,&character);

				//handle the short push duration
				sprintf (params, "%d", b->param1);
				add_parameter_to_sysfs_buffer(sysfs_buffer,params);

				//handle the long push duration
				sprintf (params, "%d", b->param2);
				add_str_to_sysfs_buffer(sysfs_buffer,params); //it's a parameter but handled without comma

				//handle the ending of the entry
				add_str_to_sysfs_buffer(sysfs_buffer,"]\n");
			}
			valid_btn++;
		}
	}
}


static void extract_key_value(char *to_parse, char *key, char *value)
{
	unsigned int len = strlen(to_parse);

	char* eq_pos = strchr(to_parse,'=');

	if (eq_pos != NULL)
	{
		strncpy(key,to_parse,(eq_pos-to_parse)); //extract key
		strncpy(value,eq_pos+1,len-(eq_pos-to_parse)-1); //extract value
	}
	else
	{
		printk("%s: key/value pair is not valid\n",MODULE_NAME);
	}
}


static int isNumber(const char *s)
{
    const char valid[] = "0123456789";
    const char *tmp = s;

    while( *tmp )
    {
        if( strchr(valid, *tmp) )
            return -1;
        tmp++;
    }
    printk("%s: parameter \"%s\" is not a number\n",MODULE_NAME,s);
    return 1;
}

static int isValidKey(const char s)
{
	char *valid_btn = valid_button_char;
	while (*valid_btn != NULL)
	{
		if (*valid_btn == s)
		{
			return 0;
		}
		valid_btn++;
	}
	printk("%s: key \"%c\" is not valid\n",MODULE_NAME,s);
	return -1;
}

static void parse_button_mapping(const char* to_buf, struct parsed_data_s *parsed_data)
{
	//new element
	struct elem_s *new_elem;

	char *user_config = to_buf;
	char* res;

	int identified_name = 0;
	while( (res=strsep(&user_config,":,")) != NULL ) //parse name and multiply key/value pairs
	{
		if (!identified_name)
		{
			//res contains the name
			identified_name=1;
			if ((parsed_data->gl_btn_index = button_idx_by_name(res)) < 0)
			{
				printk("%s: button is unknown\n",MODULE_NAME);
				break;
			}
			else
			{
				char **valid_btn = valid_buttons;
				while (*valid_btn != NULL)
				{
					if (strcmp(*valid_btn,res) == 0)
					{
						strncpy(parsed_data->name,res,strlen(res));
						//printk("button is known: %s\n",parsed_data->name);
					}
					else
					{
						printk("%s: You are not allowed to change %s\n",MODULE_NAME,res);
					}
					valid_btn++;
				}
			}
		}
		else
		{
			//this is the parameter list to be configured
			char key[MAX_KEY_VALUE_SIZE] = {0}, value[MAX_KEY_VALUE_SIZE] = {0};
			extract_key_value(res,key,value);

			//check if parsed key is someone of list valid_keys
			char **valid_k = valid_keys;
			while (*valid_k != NULL)
			{
				//if key is valid, add it to parsed data
				if (strcmp(*valid_k,key) == 0)
				{
					//create new container for list element key/value
					new_elem = (struct elem_s *)kzalloc(sizeof(struct elem_s),GFP_KERNEL);

					//copy data to container
					strncpy(new_elem->key,key,MAX_KEY_VALUE_SIZE);
					strncpy(new_elem->value,value,MAX_KEY_VALUE_SIZE);

					//add the new item to list
					list_add(&(new_elem->list), &(parsed_data->elem.list));
					break;
				}
				valid_k++;
			}
			if (*valid_k == NULL)
			{
				printk("%s: Parameter %s is not known\n",MODULE_NAME,key);
			}
		}
	}
}

static void free_parse_button_mapping(struct parsed_data_s *parsed_data)
{
	struct elem_s *new_elem;
	struct list_head *pos, *q;
	new_elem = (struct elem_s *)kzalloc(sizeof(struct elem_s),GFP_KERNEL);

	list_for_each_safe(pos, q, &(parsed_data->elem.list)){
			 new_elem= list_entry(pos, struct elem_s, list);
			 list_del(pos);
			 kfree(new_elem);
	}
}

static void init_parse_button_mapping(struct parsed_data_s *parsed_data)
{
	INIT_LIST_HEAD(&(parsed_data->elem.list));
	memset(parsed_data->name,0,sizeof(parsed_data->name)/sizeof(parsed_data->name[0]));
}

static void change_button_mapping(struct parsed_data_s *parsed_data)
{
	struct elem_s *new_elem;
	struct list_head *pos;
	unsigned long res = 0;
	new_elem = (struct elem_s *)kzalloc(sizeof(struct elem_s),GFP_KERNEL);

	list_for_each(pos, &(parsed_data->elem.list))
	{
		new_elem = list_entry(pos, struct elem_s, list);

		if ((strcmp(new_elem->key,"c1") == 0) && (isValidKey(new_elem->value[0])==0))
		{
			global_buttons[parsed_data->gl_btn_index].c1 = new_elem->value[0];
		}
		else if ((strcmp(new_elem->key,"c2") == 0) && (isValidKey(new_elem->value[0])==0))
		{
			global_buttons[parsed_data->gl_btn_index].c2 = new_elem->value[0];
		}
		else if ((strcmp(new_elem->key,"param1") == 0) && isNumber(new_elem->value))
		{
			if (strict_strtoul((new_elem->value),10,&res))
			{
				printk("Can not convert value of %s\n",new_elem->key);
			}
			else
			{
				global_buttons[parsed_data->gl_btn_index].param1 = (int)res;
			}
		}
		else if ((strcmp(new_elem->key,"param2") == 0) && isNumber(new_elem->value))
		{
			if (strict_strtoul((new_elem->value),10,&res))
			{
				printk("Can not convert value of %s\n",new_elem->key);
			}
			else
			{
				global_buttons[parsed_data->gl_btn_index].param2 = (int)res;
			}
		}

	}
}

//Called on read, user access (sysfs)
//notice: buf will be allocated with PAGE_SIZE
static ssize_t show_button_mapping(struct device* dev, struct device_attribute* attr, const char* buf)
{
	// fill the buffer, return the buffer size
	scnprintf(buf,sysfs_buffer_size,"%s",sysfs_buffer);
	return sysfs_buffer_size;
}


//Called on write, user access (sysfs)
//notice: buf will be allocated with PAGE_SIZE
//Example: echo wifi_wps:c1=x,c2=y,param1=234,param2=456 > /sys/class/buttons/dvlbutton/button_mapping
static ssize_t store_button_mapping(struct device* dev, struct device_attribute* attr, const char* buf, size_t count)
{
	struct parsed_data_s parsed_data;
	init_parse_button_mapping(&parsed_data);

	//parse user input
	parse_button_mapping(buf,&parsed_data);

	//change the mapping in connection with the parsed data
	change_button_mapping(&parsed_data);

	//update global buffer which contains the actual mapping
	update_global_button_list_for_sysfs();

	//to avoid memory leaks
	free_parse_button_mapping(&parsed_data);

	return count;
}


//Declare the sysfs entries. The macros create instances of dev_attr_button_mapping
DEVICE_ATTR(button_mapping,0644,show_button_mapping,store_button_mapping);
#endif


//
// dvlbutton
// 

/*
 * This function is called in select(), poll() etc
 */
static unsigned int dvlbutton_poll(struct file *filp, poll_table *wait) {
    struct dvlbutton_dev *dev = filp->private_data;
    unsigned int mask = 0;

	unsigned int read_pos;
	if ((unsigned int) (filp->f_pos) < file_offset)
	{
		printk(KERN_WARNING "Requested button events are not available anymore\n");
		filp->f_pos = file_offset;
	}
	read_pos = (unsigned int) (filp->f_pos) - file_offset;

    down(&dev->sem);
    poll_wait(filp, &dev->inq, wait);
    if (read_pos < write_pos)
        mask |= POLLIN | POLLRDNORM; /* readable */

    up(&dev->sem);
    return mask;
}

/*
 * Called if read() is called
 */
static ssize_t dvlbutton_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos) {
    struct dvlbutton_dev *dev = filp->private_data;
    ssize_t retval = 0;

	unsigned int read_pos;
	if ((unsigned int) (filp->f_pos) < file_offset)
	{
		printk(KERN_WARNING "Requested button events are not available anymore\n");
		filp->f_pos = file_offset;
	}
	read_pos = (unsigned int) (filp->f_pos) - file_offset;

    if (down_interruptible(&dev->sem))
        return -ERESTARTSYS;
        
	while (!(filp->f_flags & O_NONBLOCK) && (read_pos >= write_pos)) { /* nothing to read */
	    up(&dev->sem);
	    if (wait_event_interruptible(dev->inq, read_pos < write_pos))
	        return -ERESTARTSYS;
	    if (down_interruptible(&dev->sem))
	        return -ERESTARTSYS;
	}

	if(!(filp->f_flags & O_NONBLOCK))
    {
		if (count >= 1)
			retval = emit_char(buf, read_pos);
		*f_pos += retval;
	}
	else
	{
		if (count >= num_buttons())
			retval = emit_states(buf);
	}
	
    up(&dev->sem);
    return retval;

}

static int dvlbutton_release(struct inode *unused_inode, struct file* unused_filp) {
    return 0;
}

/*
 * Called if open() is called
 */
static int dvlbutton_open(struct inode *inode, struct file* filp) {
    struct dvlbutton_dev *dev;

    dev = container_of(inode->i_cdev, struct dvlbutton_dev, cdev);
    filp->private_data = dev;
    filp->f_pos = write_pos + file_offset;
    return 0;
}

/*
 * Called if ioctl() is called
 */
static int dvlbutton_ioctl(struct inode *unused_inode, struct file *unused_filp, unsigned int cmd, unsigned long arg) {
    int retval = 0;

    switch (cmd) {
    case DVLBUTTON_IOCRESET: {
    	int i;
    	for (i = 0; i < num_buttons(); i++)
    		global_contexts[i].is_long = 0;
    }
        break;
    case DVLBUTTON_IOCGBTNSTATE: {
        int idx = button_idx(arg);
        if (idx >= 0 && global_contexts[idx].last_change == 1) retval = 1;
        else if (idx < 0) retval = -1;
    }
        break;
    default:
        return -ENOTTY;
    }
    return retval;
}

static struct file_operations dvlbutton_fops = {
    .owner = THIS_MODULE,
    .open = dvlbutton_open,
    .release = dvlbutton_release,
    .read = dvlbutton_read,
    .poll = dvlbutton_poll,
    .ioctl = dvlbutton_ioctl,
};

/*
 * Setup character devices
 */
static void dvlbutton_setup_cdev(struct dvlbutton_dev *dev, unsigned int index) {
    int err;
    unsigned int devno = MKDEV(dvlbutton_major, dvlbutton_minor + index);

    cdev_init(&dev->cdev, &dvlbutton_fops);
    dev->cdev.owner = THIS_MODULE;
    dev->cdev.ops = &dvlbutton_fops;
    err = cdev_add(&dev->cdev, devno, 1);
    init_MUTEX(&dev->sem);
    init_waitqueue_head(&dev->inq);
    if (err){
        printk(KERN_DEBUG "Error %d adding %s%d",err,MODULE_NAME,index);
    }


//TODO: place to dvlbutton_setup_sysfs and handle error with return
#ifdef _DVL_ENABLE_BUTTON_MAPPING
	buttons_class = class_create(THIS_MODULE, "buttons");
	if (IS_ERR(buttons_class))
	{
		printk("sysfs error: Can not create class\n");
	}
	else
	{
		buttons_device = device_create(buttons_class, NULL, devno, NULL, MODULE_NAME);
		if (IS_ERR(buttons_device))
		{
			printk("failed to create device buttons\n");
		    class_unregister(buttons_class);
		    class_destroy(buttons_class);
		}
		else
		{
			int retval = device_create_file(buttons_device, &dev_attr_button_mapping);
			if (retval < 0)
			{
				printk("failed to create write /sys endpoint - continuing without\n");
			}
		}
	}
#endif

}

/*
 * GPIOs
 */
static void dvl_init_all_gpios(void)
{
	dvlbutton_entry_t *b;

	dvl_gpio_init();

	for (b = global_buttons; b->type != TYPE_NONE; b++)
	{
		if (b->gpio >= 0)
		{
			dvl_gpio_line_config_in(b);
		}
	}
}

/*
 * IRQs
 */

static int dvl_init_all_irqs(void)
{
	int rc = 0;
	dvlbutton_entry_t *b, *c;

	for (b = global_buttons; b->type != TYPE_NONE; b++)
	{
		int irq = b->irq;
		for (c = global_buttons; c != b; c++)
		{
			if (c->irq == b->irq)
				irq = -1;
		}
		
		if (irq >= 0)
		{
			rc = request_irq(
				(unsigned int) irq,
				irq_handler,
#if defined(IRQF_DISABLED)
                IRQF_DISABLED,
#elif defined(SA_INTERRUPT)
				SA_INTERRUPT,
#else
				0,
#endif
				"devolo button handler",
				&global_dev
			);
			
			printk(KERN_INFO "dvlbutton: IRQ %d requested, %s\n", irq, rc < 0 ? "failed" : "successful");
			if (rc < 0)
				return rc;
		}
	}
	
	return rc;
}

static void dvl_shutdown_all_irqs(void)
{
	dvlbutton_entry_t *b, *c;
	
	for (b = global_buttons; b->type != TYPE_NONE; b++)
	{
		int irq = b->irq;
		for (c = global_buttons; c != b; c++)
		{
			if (c->irq == b->irq)
				irq = -1;
		}
		
		if (irq >= 0)
		{
			free_irq((unsigned int) irq, &global_dev);
			printk(KERN_INFO "IRQ %d freed\n", irq);
		}
	}
}


/*
 * Init module after insmod or modprobe
 */

static int __init init_mod(void) {
	int result = 0;
    int num = 0;
    int i;

    dev_id = MKDEV(dvlbutton_major, dvlbutton_minor);
    result = register_chrdev_region(dev_id,1,MODULE_NAME);
    if (result < 0) {
        printk(KERN_WARNING "%s, can't get major %d\n",MODULE_NAME, dvlbutton_major);
        return result;
    }
    dvlbutton_setup_cdev(&global_dev, 0);

	num = num_buttons();

	global_contexts = kcalloc((unsigned int) num, sizeof(struct dvlbutton_context), GFP_KERNEL);
	if (global_contexts == NULL)
	{
		printk(KERN_ERR "dvlbutton: unable to allocate memory\n");
		return -ENOMEM;
	}
	for (i = 0; i < num; i++)
	{
		global_contexts[i].last_change = -1;
		ktime_get_ts(&(global_contexts[i].last_time));
		global_contexts[i].dev = &global_dev;
	}

	if (dvl_init_all_irqs() < 0)
	{
		printk(KERN_DEBUG "dvlbutton: Button IRQ not available\n");
		cdev_del(&global_dev.cdev);
		unregister_chrdev_region(dev_id,1);
#ifdef _DVL_ENABLE_BUTTON_MAPPING
        if (buttons_device != NULL) {
            device_remove_file(buttons_device, &dev_attr_button_mapping);
        }
        if (buttons_class != NULL) {
            device_destroy(buttons_class, MKDEV(dvlbutton_major, dvlbutton_minor));
            class_unregister(buttons_class);
            class_destroy(buttons_class);
        }
#endif
		return -EIO;
	}

	dvl_init_all_gpios();
	dvl_gpio_clear();
    
    printk(KERN_INFO "dvlbutton: Module %s initialized\n", MODULE_NAME);

#ifdef _DVL_ENABLE_BUTTON_MAPPING
	//look at dvlbuttons_setup_cdev
	update_global_button_list_for_sysfs();
#endif

    return 0;
}

/*
 * Cleanup after rmmod
 */
static void __exit exit_mod(void) {
	int i;
	
    dvl_gpio_shutdown();
    dvl_shutdown_all_irqs();
    cdev_del(&global_dev.cdev);
    unregister_chrdev_region(dev_id,1);
    
    for (i = 0; i < num_buttons(); i++)
    	del_timer_sync(&(global_contexts[i].timer));
    
    if (global_contexts != NULL)
    	kfree(global_contexts);
    
#ifdef _DVL_ENABLE_BUTTON_MAPPING
    device_remove_file(buttons_device, &dev_attr_button_mapping);
    device_destroy(buttons_class, MKDEV(dvlbutton_major, dvlbutton_minor));
    class_unregister(buttons_class);
    class_destroy(buttons_class);
#endif

    printk(KERN_INFO "Module %s removed\n", MODULE_NAME);
}

MODULE_AUTHOR("Christian Taedcke / Christian Petry");
MODULE_DESCRIPTION("Button module");

module_param(dvlbutton_major,uint,S_IRUGO);
module_param(dvlbutton_minor,uint,S_IRUGO);
module_init(init_mod);
module_exit(exit_mod);
