/*
 * Copyright (C) 2010 STMicroelectronics Ltd
 * Author(s): Filippo Arcidiacono <filippo.arcidiacono@st.com>
 *
 * Licensed under the LGPL v2.1, see the file COPYING.LIB in this tarball.
 *
 */

#ifdef __NR_socketcall
extern int __socketcall(int call, unsigned long *args) attribute_hidden;

/* Various socketcall numbers */
#define SYS_SOCKET      1
#define SYS_BIND        2
#define SYS_CONNECT     3
#define SYS_LISTEN      4
#define SYS_ACCEPT      5
#define SYS_GETSOCKNAME 6
#define SYS_GETPEERNAME 7
#define SYS_SOCKETPAIR  8
#define SYS_SEND        9
#define SYS_RECV        10
#define SYS_SENDTO      11
#define SYS_RECVFROM    12
#define SYS_SHUTDOWN    13
#define SYS_SETSOCKOPT  14
#define SYS_GETSOCKOPT  15
#define SYS_SENDMSG     16
#define SYS_RECVMSG     17
#define SYS_ACCEPT4		18

#if defined __ASSUME_NOT_MULTIPLEXED_SOCKETFUNCS && \
	defined __UCLIBC_HAS_SOCKETCALLS_COMPATIBILITY__
/*
 * If backward compatibility is required, all the socket functions based on the
 * socketcall multiplexer will be defined with the __compat_ prefix, and used
 * as fallback in the wrapper when the syscall fails and errno is ENOSYS.
 */
#define SOCKETCALLS_CANCEL_FUNC(__name) static __concat(__compat_, __name)
#define SOCKETCALLS_FUNC(__name) SOCKETCALLS_CANCEL_FUNC(__name)
/* Do not handle cancellation when used as fallback implementation */
#define SOCKETCALLS_CANCEL_CODE   0
#else

/*
 * Not multiplexed socket functions are not available, so just use the socketcall
 * multiplexer. Cancellable versions are named with the __libc_ prefix.
 */
#define SOCKETCALLS_CANCEL_FUNC(__name) __concat(__libc_, __name)
#define SOCKETCALLS_FUNC(__name) __concat(, __name)
#define SOCKETCALLS_CANCEL_CODE   1
#endif

#ifdef L_accept
int  SOCKETCALLS_CANCEL_FUNC(accept)(int s, struct sockaddr *addr,
									socklen_t * addrlen)
{
	unsigned long args[3];

	args[0] = s;
	args[1] = (unsigned long) addr;
	args[2] = (unsigned long) addrlen;

#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return __socketcall(SYS_ACCEPT, args);

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_ACCEPT, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return __socketcall(SYS_ACCEPT, args);
#endif
}
#endif

#ifdef L_accept4
int  SOCKETCALLS_CANCEL_FUNC(accept4)(int fd, struct sockaddr *addr,
										socklen_t *addrlen, int flags)
{
	unsigned long args[4];

	args[0] = fd;
	args[1] = (unsigned long) addr;
	args[2] = (unsigned long) addrlen;
	args[3] = flags;
#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return __socketcall(SYS_ACCEPT4, args);
#ifdef __UCLIBC_HAS_THREADS_NATIVE__
		int oldtype = LIBC_CANCEL_ASYNC ();
		int result = __socketcall(SYS_ACCEPT4, args);
		LIBC_CANCEL_RESET (oldtype);
		return result;
#endif
#else
	return __socketcall(SYS_ACCEPT4, args);
#endif

}
#endif

#ifdef L_bind

int SOCKETCALLS_FUNC(bind)(int sockfd, const struct sockaddr *myaddr,
							socklen_t addrlen)
{
	unsigned long args[3];

	args[0] = sockfd;
	args[1] = (unsigned long) myaddr;
	args[2] = addrlen;
	return __socketcall(SYS_BIND, args);
}
#endif

#ifdef L_connect
int SOCKETCALLS_CANCEL_FUNC(connect)(int sockfd, const struct sockaddr *saddr,
									socklen_t addrlen)
{
	unsigned long args[3];

	args[0] = sockfd;
	args[1] = (unsigned long) saddr;
	args[2] = addrlen;

#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return __socketcall(SYS_CONNECT, args);

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_CONNECT, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return __socketcall(SYS_CONNECT, args);
#endif
}
#endif

#ifdef L_getpeername
int SOCKETCALLS_FUNC(getpeername)(int sockfd, struct sockaddr *addr,
								socklen_t * paddrlen)
{
	unsigned long args[3];

	args[0] = sockfd;
	args[1] = (unsigned long) addr;
	args[2] = (unsigned long) paddrlen;
	return __socketcall(SYS_GETPEERNAME, args);
}
#endif

#ifdef L_getsockname
int SOCKETCALLS_FUNC(getsockname)(int sockfd, struct sockaddr *addr,
								socklen_t * paddrlen)
{
	unsigned long args[3];

	args[0] = sockfd;
	args[1] = (unsigned long) addr;
	args[2] = (unsigned long) paddrlen;
	return __socketcall(SYS_GETSOCKNAME, args);
}
#endif

#ifdef L_getsockopt
int SOCKETCALLS_FUNC(getsockopt)(int fd, int level, int optname, __ptr_t optval,
								socklen_t * optlen)
{
	unsigned long args[5];

	args[0] = fd;
	args[1] = level;
	args[2] = optname;
	args[3] = (unsigned long) optval;
	args[4] = (unsigned long) optlen;
	return (__socketcall(SYS_GETSOCKOPT, args));
}
#endif

#ifdef L_listen
int SOCKETCALLS_FUNC(listen)(int sockfd, int backlog)
{
	unsigned long args[2];

	args[0] = sockfd;
	args[1] = backlog;
	return __socketcall(SYS_LISTEN, args);
}
#endif

#ifdef L_recv
ssize_t SOCKETCALLS_CANCEL_FUNC(recv)(int sockfd, __ptr_t buffer, size_t len,
									int flags)
{
	unsigned long args[4];

	args[0] = sockfd;
	args[1] = (unsigned long) buffer;
	args[2] = len;
	args[3] = flags;

#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return (__socketcall(SYS_RECV, args));

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_RECV, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return (__socketcall(SYS_RECV, args));
#endif
}
#endif

#ifdef L_recvfrom
ssize_t SOCKETCALLS_CANCEL_FUNC(recvfrom)(int sockfd, __ptr_t buffer,
										size_t len, int flags,
										struct sockaddr *to, socklen_t * tolen)
{
	unsigned long args[6];

	args[0] = sockfd;
	args[1] = (unsigned long) buffer;
	args[2] = len;
	args[3] = flags;
	args[4] = (unsigned long) to;
	args[5] = (unsigned long) tolen;

#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return (__socketcall(SYS_RECVFROM, args));

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_RECVFROM, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return (__socketcall(SYS_RECVFROM, args));
#endif
}
#endif

#ifdef L_recvmsg
ssize_t SOCKETCALLS_CANCEL_FUNC(recvmsg)(int sockfd, struct msghdr *msg,
										int flags)
{
	unsigned long args[3];

	args[0] = sockfd;
	args[1] = (unsigned long) msg;
	args[2] = flags;

#if	SOCKETCALLS_CANCEL_CODE
if (SINGLE_THREAD_P)
	return (__socketcall(SYS_RECVMSG, args));

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_RECVMSG, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return (__socketcall(SYS_RECVMSG, args));
#endif
}
#endif

#ifdef L_send
ssize_t SOCKETCALLS_CANCEL_FUNC(send)(int sockfd, const void *buffer,
									size_t len, int flags)
{
	unsigned long args[4];

	args[0] = sockfd;
	args[1] = (unsigned long) buffer;
	args[2] = len;
	args[3] = flags;

#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return (__socketcall(SYS_SEND, args));

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_SEND, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return (__socketcall(SYS_SEND, args));
#endif
}
#endif

#ifdef L_sendmsg
ssize_t SOCKETCALLS_CANCEL_FUNC(sendmsg)(int sockfd, const struct msghdr *msg,
										int flags)
{
	unsigned long args[3];

	args[0] = sockfd;
	args[1] = (unsigned long) msg;
	args[2] = flags;

#if	SOCKETCALLS_CANCEL_CODE
if (SINGLE_THREAD_P)
	return (__socketcall(SYS_SENDMSG, args));

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_SENDMSG, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return (__socketcall(SYS_SENDMSG, args));
#endif
}
#endif

#ifdef L_sendto
ssize_t SOCKETCALLS_CANCEL_FUNC(sendto)(int sockfd,
								const void *buffer, size_t len, int flags,
								const struct sockaddr *to, socklen_t tolen)
{
	unsigned long args[6];

	args[0] = sockfd;
	args[1] = (unsigned long) buffer;
	args[2] = len;
	args[3] = flags;
	args[4] = (unsigned long) to;
	args[5] = tolen;

#if	SOCKETCALLS_CANCEL_CODE
	if (SINGLE_THREAD_P)
		return (__socketcall(SYS_SENDTO, args));

#ifdef __UCLIBC_HAS_THREADS_NATIVE__
	int oldtype = LIBC_CANCEL_ASYNC ();
	int result = __socketcall(SYS_SENDTO, args);
	LIBC_CANCEL_RESET (oldtype);
	return result;
#endif
#else
	return (__socketcall(SYS_SENDTO, args));
#endif
}
#endif

#ifdef L_setsockopt
int SOCKETCALLS_FUNC(setsockopt)(int fd, int level, int optname,
								const void *optval, socklen_t optlen)
{
	unsigned long args[5];

	args[0] = fd;
	args[1] = level;
	args[2] = optname;
	args[3] = (unsigned long) optval;
	args[4] = optlen;
	return (__socketcall(SYS_SETSOCKOPT, args));
}
#endif

#ifdef L_shutdown
int SOCKETCALLS_FUNC(shutdown)(int sockfd, int how)
{
	unsigned long args[2];

	args[0] = sockfd;
	args[1] = how;
	return (__socketcall(SYS_SHUTDOWN, args));
}
#endif

#ifdef L_socket
int SOCKETCALLS_FUNC(socket)(int family, int type, int protocol)
{
	unsigned long args[3];

	args[0] = family;
	args[1] = type;
	args[2] = (unsigned long) protocol;
	return __socketcall(SYS_SOCKET, args);
}
#endif

#ifdef L_socketpair
int SOCKETCALLS_FUNC(socketpair)(int family, int type, int protocol,
								int sockvec[2])
{
	unsigned long args[4];

	args[0] = family;
	args[1] = type;
	args[2] = protocol;
	args[3] = (unsigned long) sockvec;
	return __socketcall(SYS_SOCKETPAIR, args);
}
#endif

#endif


