//#include <linux/config.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/resource.h>

#include <linux/console.h>
#include <asm/serial.h>

#include <linux/tty.h>
#include <linux/time.h>
#include <linux/serial_core.h>
#include <linux/serial.h>
#include <linux/serial_8250.h>
#include <linux/miscdevice.h>

#include <asm/mach-atheros/atheros.h>
#include <asm/delay.h>
#include <linux/proc_fs.h>


#ifndef CONFIG_WATCHDOG_RESET_TIMER

#define ATH_DEFAULT_WD_TMO	(20ul * USEC_PER_SEC)

#define FACTORY_RESET		0x89ABCDEF

#define ATH_GPIO_RESET	21

#ifdef ATH_WDT_TEST_CODE
#	define wddbg printk
#else
#	define wddbg(junk, ...)
#endif /* ATH_WDT_TEST_CODE 8 */

extern uint32_t ath_ahb_freq;

typedef struct {
	int open:1, can_close:1, tmo, action;
	wait_queue_head_t wq;
} ath_wdt_t;

static ath_wdt_t wdt_softc_array;

static ath_wdt_t *wdt = &wdt_softc_array;

irqreturn_t ath_wdt_isr(int, void *);

#ifdef ATH_WDT_TEST_CODE
/* Return the value present in the watchdog register */
static inline uint32_t ath_get_wd_timer(void)
{
	uint32_t val;

	val = (uint32_t) ath_reg_rd(ATH_WATCHDOG_TMR);
	val = (val * USEC_PER_SEC) / ath_ahb_freq;

	return val;
}
#endif /* ATH_WDT_TEST_CODE */

/* Set the timeout value in the watchdog register */
void ath_set_wd_timer(uint32_t usec /* micro seconds */)
{
#ifdef CONFIG_MACH_AR934x
	usec = usec * (ath_ref_freq / USEC_PER_SEC);
#else
	usec = usec * (ath_ahb_freq / USEC_PER_SEC);
#endif

	wddbg("%s: 0x%08x\n", __func__, usec);

	ath_reg_wr(ATH_WATCHDOG_TMR, usec);
}

int ath_set_wd_timer_action(uint32_t val)
{
	if (val & ~ATH_WD_ACT_MASK) {
		return EINVAL;
	}

	wdt->action = val;

	/*
	 * bits  : 31 30 - 2 0-1
	 * access: RO  rsvd  Action
	 *
	 * Since bit 31 is read only and rest of the bits
	 * are zero, don't have to do a read-modify-write
	 */
	ath_reg_wr(ATH_WATCHDOG_TMR_CONTROL, val);
	return 0;
}

#ifdef ATH_WDT_TEST_CODE
static inline uint32_t ath_get_wd_timer_action(void)
{
	return (uint32_t) (ath_reg_rd(ATH_WATCHDOG_TMR_CONTROL) &
			   ATH_WD_ACT_MASK);
}

static inline uint32_t ath_get_wd_timer_last(void)
{
	return ((uint32_t) (ath_reg_rd(ATH_WATCHDOG_TMR_CONTROL) &
			    ATH_WD_LAST_MASK) >> ATH_WD_LAST_SHIFT);
}
#endif /* ATH_WDT_TEST_CODE */

irqreturn_t ath_wdt_isr(int cpl, void *dev_id)
{
	unsigned delay;
	extern int ath_gpio_in_val(int);

#define UDELAY_COUNT 4000

	wddbg("%s: invoked\n", __func__);

	for (delay = UDELAY_COUNT; delay; delay--) {
		if (ath_gpio_in_val(ATH_GPIO_RESET)) {
			break;
		}
		udelay(1000);
	}

	wddbg("%s: %d", __func__, delay);

	if (!delay) {
		wake_up(&wdt->wq);
	} else {
		extern void ath_restart(char *);
		ath_restart(NULL);
	}
	return IRQ_HANDLED;
}

static int athwdt_open(struct inode *inode, struct file *file)
{
	wddbg("%s: called\n", __func__);

	if (MINOR(inode->i_rdev) != WATCHDOG_MINOR) {
		return -ENODEV;
	}

	if (wdt->open) {
		return -EBUSY;
	}

	wdt->open = 1;
	wdt->tmo = ATH_DEFAULT_WD_TMO;
	wdt->action = ATH_WD_ACT_NONE;
	wdt->can_close = 0;
	init_waitqueue_head(&wdt->wq);

	ath_set_wd_timer(wdt->tmo);
	ath_set_wd_timer_action(ATH_WD_ACT_NONE);

	return nonseekable_open(inode, file);
}

static int athwdt_close(struct inode *inode, struct file *file)
{
	wddbg("%s: called\n", __func__);

	if (MINOR(inode->i_rdev) != WATCHDOG_MINOR) {
		return -ENODEV;
	}

	if (!wdt->can_close) {
		wddbg("%s: clearing action\n", __func__);
		ath_set_wd_timer_action(ATH_WD_ACT_NONE);
	} else {
		wddbg("%s: not clearing action\n", __func__);
	}
	wdt->open = 0;
	return 0;
}

static ssize_t
athwdt_read(struct file *file, char *buf, size_t count, loff_t * ppos)
{
	wddbg("%s: called\n", __func__);

	return -ENOTSUPP;
}

static int
athwdt_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg)
{
	int ret = 0;

	wddbg("%s: called\n", __func__);

	switch (cmd) {
	case FACTORY_RESET:
		wddbg("%s: intr action\n", __func__);

		if ((ret = request_irq(ATH_MISC_IRQ_WATCHDOG,
				       ath_wdt_isr,
				       0, "Watchdog Timer", wdt))) {
			wddbg("%s: request_irq %d\n", __func__, ret);
			return ret;
		}

		ath_set_wd_timer_action(ATH_WD_ACT_GP_INTR);
		sleep_on(&wdt->wq);
		free_irq(ATH_MISC_IRQ_WATCHDOG, wdt);
		break;

	default:
		ret = -EINVAL;
	}

	return ret;
}

static ssize_t
athwdt_write(struct file *file, const char *buf, size_t count, loff_t * ppos)
{
	int i;
	char c;

	wddbg("%s: called\n", __func__);

	for (i = 0; i != count; i++) {
		if (get_user(c, buf + i)) {
			return -EFAULT;
		}

		if (c == 'V') {
			wdt->can_close = 1;
			break;
		}
	}

	if (i) {
		ath_set_wd_timer(wdt->tmo);
		return 1;
	}

	return 0;
}

static struct file_operations athwdt_fops = {
      read:athwdt_read,
      write:athwdt_write,
      ioctl:athwdt_ioctl,
      open:athwdt_open,
      release:athwdt_close
};

static struct miscdevice athwdt_miscdev = {
	WATCHDOG_MINOR,
	"watchdog",
	&athwdt_fops
};


static void athwdt_timer_action(unsigned long dummy);

static DEFINE_TIMER(athwdt_timer, athwdt_timer_action, 0, 0);
static struct proc_dir_entry *panic_entry;


#define ATHWDT_WATCHDOT_TIMER_DEFAULT 	(10000000)                          /* 10s */
#define ATHWDT_KERNEL_TIMER             (jiffies + ((1000) * HZ) / 1000)    /*  1s */

static void athwdt_timer_action(unsigned long dummy)
{
	ath_set_wd_timer(ATHWDT_WATCHDOT_TIMER_DEFAULT);
	ath_set_wd_timer_action(ATH_WD_ACT_RESET);
	mod_timer(&athwdt_timer, ATHWDT_KERNEL_TIMER);
}

int debug_panic_proc_write(struct file *file, const char *buffer, unsigned long count, void *data)
{
	panic("debug panic!\n");
	return(count);
}

static int debug_panic_event(struct notifier_block *blk, unsigned long event, void *ptr)
{
#if 1
	void (*p)(void);

	p = 0xbd00009c;
	/* p = 0xbfc00000; */

	(*p)();
#endif

	printk(KERN_EMERG "%s:%d: here jiffies:%llu\n", __func__, __LINE__, jiffies);
	del_timer(&athwdt_timer);
	ath_set_wd_timer(1000000);
	ath_set_wd_timer_action(ATH_WD_ACT_RESET);
	return NOTIFY_DONE;
}

static struct notifier_block debug_panic_block = {
	debug_panic_event,
	NULL,
	INT_MAX /* try to do it first */
};


static void athwdt_timer_init(void)
{
	uint32_t* p;
	printk(KERN_EMERG "%s:%d: here jiffies:%llu\n", __func__, __LINE__, jiffies);
	ath_set_wd_timer(ATHWDT_WATCHDOT_TIMER_DEFAULT * 6);
	ath_set_wd_timer_action(ATH_WD_ACT_RESET);
	mod_timer(&athwdt_timer, ATHWDT_KERNEL_TIMER);
	panic_entry = create_proc_entry("debug_panic", 0644, NULL);
	panic_entry->nlink = 1;
	panic_entry->write_proc = debug_panic_proc_write;

	atomic_notifier_chain_register(&panic_notifier_list,
	                &debug_panic_block);
	
}

int __init athwdt_init(void)
{
	int ret;
	extern void ath_gpio_config_input(int);

	printk("%s: Registering WDT ", __func__);
	if ((ret = misc_register(&athwdt_miscdev))) {
		printk("failed %d\n", ret);
		return ret;
	} else {
		printk("success\n");
	}

	athwdt_timer_init();

	ath_gpio_config_input(ATH_GPIO_RESET);

	return 0;
}

#else
#include <linux/proc_fs.h>
#include <linux/notifier.h>

/* Set the timeout value in the watchdog register */
//static inline void ath_set_wd_timer(uint32_t usec /* micro seconds */ )
void ath_set_wd_timer(uint32_t usec /* micro seconds */ )
{
        usec = usec * (ath_ahb_freq / USEC_PER_SEC);

        ath_reg_wr(ATH_WATCHDOG_TMR, usec);
}

//static inline int ath_set_wd_timer_action(uint32_t val)
int ath_set_wd_timer_action(uint32_t val)
{
        if (val & ~ATH_WD_ACT_MASK) {
                return EINVAL;
        }

        /*
         * bits  : 31 30 - 2 0-1
         * access: RO  rsvd  Action
         *
         * Since bit 31 is read only and rest of the bits
         * are zero, don't have to do a read-modify-write
         */
        ath_reg_wr(ATH_WATCHDOG_TMR_CONTROL, val);
        return 0;
}


static void athwdt_timer_action(unsigned long dummy);

static DEFINE_TIMER(athwdt_timer, athwdt_timer_action, 0, 0);


#define ATHWDT_WATCHDOT_TIMER_DEFAULT   (1500000)                          /* 1.5s */
#define ATHWDT_KERNEL_TIMER             (jiffies + ((1000) * HZ) / 1000)    /*  1s */

static void athwdt_timer_action(unsigned long dummy)
{
	ath_set_wd_timer(ATHWDT_WATCHDOT_TIMER_DEFAULT);
	mod_timer(&athwdt_timer, ATHWDT_KERNEL_TIMER);
}

static int athwdt_panic_event(struct notifier_block *blk, unsigned long event, void *ptr)
{
	del_timer(&athwdt_timer);
	ath_set_wd_timer(1);
	return NOTIFY_DONE;
}

static struct notifier_block athwdt_panic_block = {
	athwdt_panic_event,
	NULL,
	INT_MAX /* try to do it first */
};

irqreturn_t athwdt_isr(int cpl, void *dev_id)
{
	extern void ath_restart(char *);

	ath_restart(NULL);

        return IRQ_HANDLED;
}


int __init athwdt_init(void)
{
	int ret;

	ath_set_wd_timer(ATHWDT_WATCHDOT_TIMER_DEFAULT * 6);
	ath_set_wd_timer_action(ATH_WD_ACT_GP_INTR);
	mod_timer(&athwdt_timer, ATHWDT_KERNEL_TIMER);

	atomic_notifier_chain_register(&panic_notifier_list, &athwdt_panic_block);

	if ((ret = request_irq(ATH_MISC_IRQ_WATCHDOG,
	                       athwdt_isr,
	                       0, "Watchdog Timer", NULL))) {
		printk("%s: request_irq %d\n", __func__, ret);
		return ret;
	}


	return 0;
}

#endif /* CONFIG_WATCHDOG_RESET_TIMER */


late_initcall(athwdt_init);
